/*--------------------------------------------------------------------------------
	FILE:		line.c
	
	PURPOSE:	Contains functions for sequencing the top and bottom rows of LEDs
				on the LED Open Sign. These are the lines of LEDs above and below
				the LED "OPEN" text on the sign.
				
	DATE:		9 September 2011, releases as open source on 22 September 2012
	
	DISCLAIMER:	This software is provided 'as-is', without any express or
				implied warranty.  In no event will the author(s) be held liable
				for any damages or loss arising from the use of this software.
				
	AUTHOR:		W.A. Smith
				startingelectronics.com
--------------------------------------------------------------------------------*/
#include <avr/io.h>
#include "line.h"
#include "utils.h"

#define TOP_1   PB1
#define TOP_2   PB0
#define TOP_3   PD6

#define LED_N   PD5

#define BOT_1   PD4
#define BOT_2   PD3
#define BOT_3   PD2

void LineRotate(unsigned char direction, int del, unsigned int times)
{
    while (times) {
        if (direction == FORWARD) {
            LineOff();
            Line1TopOn();
            Line1BotOn();
            Delay(del);
            LineOff();
            Line2TopOn();
            Line2BotOn();
            Delay(del);
            LineOff();
            Line3TopOn();
            Line3BotOn();
            Delay(del);
        }
        else {
            LineOff();
            Line3TopOn();
            Line3BotOn();
            Delay(del);
            LineOff();
            Line2TopOn();
            Line2BotOn();
            Delay(del);
            LineOff();
            Line1TopOn();
            Line1BotOn();
            Delay(del);
        }
        times--;
    }
}

void DualLineRotate(unsigned char direction, int del, unsigned int times)
{
    while (times) {
        if (direction == FORWARD) {
            LineOn(LINE_BOTH);
            Line3TopOff();
            Line3BotOff();
            Delay(del);
            LineOn(LINE_BOTH);
            Line2TopOff();
            Line2BotOff();
            Delay(del);
            LineOn(LINE_BOTH);
            Line1TopOff();
            Line1BotOff();
            Delay(del);
        }
        else {
            LineOn(LINE_BOTH);
            Line1TopOff();
            Line1BotOff();
            Delay(del);
            LineOn(LINE_BOTH);
            Line2TopOff();
            Line2BotOff();
            Delay(del);
            LineOn(LINE_BOTH);
            Line3TopOff();
            Line3BotOff();
            Delay(del);
        }
        times--;
    }
}

void FlashLines(int del, unsigned char times)
{
    while (times) {
        LineOn(LINE_BOTH);
        Delay(del);
        LineOff();
        Delay(del);
        times--;
    }
}

void LineToggle(int del, unsigned char times)
{
    while (times) {
        LineOff();
        LineOn(LINE_TOP);
        Delay(del);
        LineOff();
        LineOn(LINE_BOTTOM);
        Delay(del);
        times--;
    }
}

void LineOn(unsigned char line)
{
    if (line == LINE_TOP) {
        Line1TopOn();
        Line2TopOn();
        Line3TopOn();
    }
    else if (line == LINE_BOTTOM) {
        Line1BotOn();
        Line2BotOn();
        Line3BotOn();
    }
    else {
        Line1TopOn();
        Line2TopOn();
        Line3TopOn();
        Line1BotOn();
        Line2BotOn();
        Line3BotOn();
    }
}

void LineOff(void)
{
    Line1TopOff();
    Line2TopOff();
    Line3TopOff();
    Line1BotOff();
    Line2BotOff();
    Line3BotOff();
}

void Line1TopOn(void)
{
    PORTB |= (1 << TOP_1);
}
void Line1TopOff(void)
{
    PORTB &= ~(1 << TOP_1);
}
void Line2TopOn(void)
{
    PORTB |= (1 << TOP_2);
}
void Line2TopOff(void)
{
    PORTB &= ~(1 << TOP_2);
}
void Line3TopOn(void)
{
    PORTD |= (1 << TOP_3);
}
void Line3TopOff(void)
{
    PORTD &= ~(1 << TOP_3);
}
void Line1BotOn(void)
{
    PORTD |= (1 << BOT_1);
}
void Line1BotOff(void)
{
    PORTD &= ~(1 << BOT_1);
}
void Line2BotOn(void)
{
    PORTD |= (1 << BOT_2);
}
void Line2BotOff(void)
{
    PORTD &= ~(1 << BOT_2);
}
void Line3BotOn(void)
{
    PORTD |= (1 << BOT_3);
}
void Line3BotOff(void)
{
    PORTD &= ~(1 << BOT_3);
}
